#include "stdafx.h"
#include "vm1.h"

C1801VM1::C1801VM1()
	: m_nClk(0)
	, m_bInit(FALSE)
{
	m_IOPins.clear();

#ifdef DBG_OUT_PLI_LOG
	m_PLILog = fopen("_PLI_log.txt", "wt");
	fprintf(m_PLILog, "        ");
	fprintf(m_PLILog, "     19     18     17     16     15     14     13     12     11     10     09     08     07     06     05     04     03     02     01     00       ");
	fprintf(m_PLILog, "     10     09     08     07     06     05     04     03     02     01     00\n");
	fprintf(m_PLILog, "        ");
	fprintf(m_PLILog, " VE-Trq IRQ3rq QBV-to IRQ2rq PSW4-T IRQ1rq ACLOup WAITmd ACLOdn DblErT QBOAto VIRQrq unused unused unused PSW7-P UnkOpc  PSW11 FBO-B4  PSW10       ");
	fprintf(m_PLILog, " RqDtRe MiCR12 RqDtRe MiCR13 RqDtRe MiCR14 FB-Pri VG-Sel VG-Sel VG-Sel VG-Sel\n");
#endif
#ifdef DBG_OUT_PLM_LOG
	m_PLMLog = fopen("_PLM_log.txt", "wt");
	fprintf(m_PLMLog, "        ");
	fprintf(m_PLMLog, "                                    33     32     31     30     29     28     27     26     25     24     23     22     21     20     19     18     17     16     15     14     13     12     11     10     09     08     07     06     05     04     03     02     01     00\n");
	fprintf(m_PLMLog, "        ");
	fprintf(m_PLMLog, "opcode  PRI T N Z V C Addr/St                                 ~mj0                               ~mj1                               ~mj2                         mj3               UncOpc                  mj4 OpType OpType OpType    mj5                               ~mj6\n");
#endif
	reg.clear();
	wire.clear();

	m_nCpuType = CPU_VM1A;

	pli_matrix_a = new vm1a_pli();
	pli_matrix_g = new vm1g_pli();
	plm_matrix_a = new vm1a_plm();
	plm_matrix_g = new vm1g_plm();

	initial();
}

C1801VM1::~C1801VM1()
{
	if (pli_matrix_a)
	{
		delete pli_matrix_a;
	}
	if (pli_matrix_g)
	{
		delete pli_matrix_g;
	}
	if (plm_matrix_a)
	{
		delete plm_matrix_a;
	}
	if (plm_matrix_g)
	{
		delete plm_matrix_g;
	}
#ifdef DBG_OUT_PLI_LOG
	fclose(m_PLILog);
#endif
#ifdef DBG_OUT_PLM_LOG
	fclose(m_PLMLog);
#endif
}

void C1801VM1::RegisterDevice(MPI *pMPI)
{
	m_nDeviceID = pMPI->RegisterDevice();
	ASSERT(m_nDeviceID != -1);
}

void C1801VM1::getMPI(MPI *pMPI) //  MPI -> CPU
{
	m_IOPins.pin_dclo_n = pMPI->dcok_n;		//input			// processor reset
	m_IOPins.pin_aclo_n = pMPI->pok_n;		//input			// power fail notificaton
	m_IOPins.pin_irq1_n = pMPI->irq1_n;		//input			// radial interrupt requests
	m_IOPins.pin_irq2_n = pMPI->irq2_n;		//input			// radial interrupt requests
	m_IOPins.pin_irq3_n = pMPI->irq3_n;		//input			// radial interrupt requests
	m_IOPins.pin_virq_n = pMPI->irq0_n;		//input			// vectored interrupt request
	m_IOPins.pin_dmgi_n = pMPI->dmgi_n;		//input			// bus granted input

	m_IOPins.pin_sync_n = pMPI->sync_n;		//inout			// address strobe
	m_IOPins.pin_ad_n   = pMPI->ad_n;		//inout[15:0]	// inverted address/data bus
	m_IOPins.pin_dout_n = pMPI->dout_n;		//inout			// data output strobe
	m_IOPins.pin_din_n  = pMPI->din_n;		//inout			// data input strobe
	m_IOPins.pin_rply_n = pMPI->rply_n;		//inout			// transaction reply
	m_IOPins.pin_dmr_n  = pMPI->dmr_n; 		//inout			// bus request shared line
	m_IOPins.pin_sack_n = pMPI->sack_n;		//inout			// bus acknowlegement
	m_IOPins.pin_init_n = pMPI->init_n;		//inout			// peripheral reset
}



void C1801VM1::setMPI(MPI *pMPI) //  CPU -> MPI
{
	//  Z,      ,     setPins
	//
	// Shared Qbus lines
	//
	pMPI->SetSYNC(m_nDeviceID, wire.pin_sync_ena, m_IOPins.pin_sync_n);		//inout			// address strobe
	pMPI->SetAD(m_nDeviceID,   wire.pin_ad_ena,   m_IOPins.pin_ad_n);		//inout[15:0]	// inverted address/data bus
	pMPI->SetDOUT(m_nDeviceID, wire.pin_ctrl_ena, m_IOPins.pin_dout_n);		//inout			// data output strobe
	pMPI->SetDIN(m_nDeviceID,  wire.pin_ctrl_ena, m_IOPins.pin_din_n);		//inout			// data input strobe
	pMPI->SetWTBT(m_nDeviceID, wire.pin_ctrl_ena, m_IOPins.pin_wtbt_n);		//output		// write/byte status
	pMPI->SetIAKO(m_nDeviceID, wire.pin_ctrl_ena, m_IOPins.pin_iako_n);		//output		// interrupt vector input

	pMPI->dmgo_n	= m_IOPins.pin_dmgo_n;									//output		// bus granted output
	//
	// "Open collector" outputs
	//
	pMPI->SetRPLY(m_nDeviceID, wire.pin_rply_out, m_IOPins.pin_rply_n);		//inout			// transaction reply
	pMPI->SetDMR(m_nDeviceID,  wire.pin_dmr_out,  m_IOPins.pin_dmr_n);		//inout			// bus request shared line
	pMPI->SetINIT(m_nDeviceID, wire.pin_init_out, m_IOPins.pin_init_n);		//inout			// peripheral reset
	pMPI->SetSACK(m_nDeviceID, wire.pin_sack_out, m_IOPins.pin_sack_n);		//inout			// bus acknowlegement
	pMPI->SetBSY(m_nDeviceID,  wire.pin_bsy_out,  m_IOPins.pin_bsy_n);		//output		// bus busy flag
	pMPI->SetSEL1(m_nDeviceID, wire.pin_sel1_out, m_IOPins.pin_sel1_n);		//output		// register select outputs
	pMPI->SetSEL2(m_nDeviceID, wire.pin_sel2_out, m_IOPins.pin_sel2_n);		//output		// register select outputs
}

/*       */
void C1801VM1::eval_p(int nClk, MPI *pMPI)
{
	m_nClk = nClk;

	if (!m_bInit)
	{
		//  .   
		m_bInit = TRUE;
		assign_all(pMPI);
		eval_all_p();
		eval_all_n();
		assign_all(pMPI);
	}

	eval_all_p();	//  always   
	assign_all(pMPI);	//  assign

}

/*       */
void C1801VM1::eval_n(int nClk, MPI *pMPI)
{
	m_nClk = nClk;


	if (!m_bInit)
	{
		//  .   
		m_bInit = TRUE;
		assign_all(pMPI);
		eval_all_n();
		eval_all_p();
		assign_all(pMPI);
	}

	eval_all_n();	//  always   
	assign_all(pMPI);	//  assign


}

/*     */
void C1801VM1::initial()
{
	//plm[33:0] = 34'o000000000000;
	for (int i = 0; i < 34; ++i)
	{
		reg.plm[i] = FALSE;
	}
	//reg.psw[15:0] = 16'o000000;
	for (int i = 0; i < 16; ++i)
	{
		reg.psw[i] = FALSE;
	}

	for (int i = 0; i < 14; ++i)
	{
		reg.gpr[i] = 0;
	}
}

/*   assign
 *    ,   assign      .
 */
void C1801VM1::assign_all(MPI *pMPI)
{
	int n = 0;
	do{
		prev_wire = wire;
		getMPI(pMPI);
		assign_in();
		assign_func();  //         .
		setMPI(pMPI);
		if (++n > 200)
		{
			//
			TRACE1("Steps num: %d\n", n);
			ASSERT(FALSE);
			break;
		}
	} while (prev_wire != wire); //       .
	//TRACE1("Steps num: %d\n", n);
}

/*   always    */
void C1801VM1::eval_all_p()
{
	snap_reg = reg; //  .      reg,   - snap_reg
	always_a(); //always   (*)
	always_posedge_pin_clk_p_or_posedge_tve_reset(); // 
	always_posedge_pin_clk_p();
	always_posedge_pin_clk_p_or_posedge_mjres_part_pin_clk_p();
	always_posedge_pin_clk_p_or_posedge_reset_part_pin_clk_p();
	always_posedge_pin_clk_p_or_posedge_abort_part_pin_clk_p();
	if (snap_reg.mjres)
	{
		always_posedge_pin_clk_n_or_posedge_mjres_part_mjres();
		always_posedge_pin_clk_p_or_posedge_mjres_part_mjres();
	}
	if (snap_reg.reset)
	{
		always_posedge_pin_clk_n_or_posedge_reset_part_reset();
		always_posedge_pin_clk_p_or_posedge_reset_part_reset();
	}
	if (snap_reg.abort)
	{
		always_posedge_pin_clk_p_or_posedge_abort_part_abort();
	}
}

/*   always    */
void C1801VM1::eval_all_n()
{
	snap_reg = reg;
	always_a();
	always_posedge_pin_clk_n();
	always_posedge_pin_clk_n_or_posedge_mjres_part_pin_clk_n();
	always_posedge_pin_clk_n_or_posedge_reset_part_pin_clk_n();
	always_posedge_pin_clk_n_or_negedge_qbus_tena();

	if (snap_reg.mjres)
	{
		always_posedge_pin_clk_n_or_posedge_mjres_part_mjres();
		always_posedge_pin_clk_p_or_posedge_mjres_part_mjres();
	}
	if (snap_reg.reset)
	{
		always_posedge_pin_clk_n_or_posedge_reset_part_reset();
		always_posedge_pin_clk_p_or_posedge_reset_part_reset();
	}
	if (snap_reg.abort)
	{
		always_posedge_pin_clk_p_or_posedge_abort_part_abort();
	}
}
/*       */
void C1801VM1::assign_in()
{
	wire.pin_pa		= (~m_IOPins.pin_pa_n) & 3;
	wire.pin_sp		= !m_IOPins.pin_sp_n;

	wire.pin_dclo	= !m_IOPins.pin_dclo_n;
	wire.pin_aclo	= !m_IOPins.pin_aclo_n;
	wire.pin_irq[1] = !m_IOPins.pin_irq1_n;
	wire.pin_irq[2] = !m_IOPins.pin_irq2_n;
	wire.pin_irq[3] = !m_IOPins.pin_irq3_n;
	wire.pin_irq[0] = !m_IOPins.pin_virq_n;

	wire.pin_init_in = !m_IOPins.pin_init_n;

	wire.pin_ad_in	= ~m_IOPins.pin_ad_n;

	wire.pin_dout_in = !m_IOPins.pin_dout_n;
	wire.pin_din_in	= !m_IOPins.pin_din_n;

	wire.pin_sync_in = !m_IOPins.pin_sync_n;

	wire.pin_rply_in = !m_IOPins.pin_rply_n;
	wire.pin_dmr_in	 = !m_IOPins.pin_dmr_n;
	wire.pin_sack_in = !m_IOPins.pin_sack_n;

	wire.pin_dmgi	= !m_IOPins.pin_dmgi_n;
}

void C1801VM1::assign_func()
{
	/*   wires     */
	BOOL bTmpB1, bTmpB2;
	BOOL bPinPA_Not_Nul = (wire.pin_pa != 0);
	//
	// External connection assignments
	//
	data2bita(wire.pin_ad_in, wire.wb_pio_dat_i, 16);
	wire.pin_init_out = reg.init_out[0] || reg.reset;
	wire.pin_ctrl_ena = reg.qbus_win || reg.iako_oe;
	wire.pin_bsy_out  = reg.sync_out || wire.pin_ctrl_ena;


	wire.au_astb  = reg.au_astb_xa || (reg.ustb && reg.au_astb_xu);
	wire.au_qstbx = reg.au_qstb_xa || (reg.ustb && reg.au_qstb_xu);

	//
	// Odd address trap detector is blocked on 1801BM1
	//
	wire.exc_oat  = FALSE;
	wire.exc_uop  = reg.plm_ena  &&  reg.plr[12];
	wire.exc_err2 = reg.plm1x_hl && !reg.plr[26];
	wire.exc_err3 = reg.plm1x_hl && !reg.plr[28];
	wire.exc_err7 = reg.plm1x_hl && !reg.plr[30];
	//______________________________________________________________________________
	//
	// Reset circuits
	//
	wire.reset_rc =  wire.pin_dclo || (wire.pin_aclo && !reg.init_out[0] && reg.init_out[2]);
	wire.abort_rc = (reg.qbus_tovf || wire.exc_oat || wire.exc_uop || reg.abort) && !(reg.abort_tm || wire.reset_rc);
	wire.mjres_rc =  wire.reset_rc || wire.abort_rc;
	//______________________________________________________________________________
	//
	// 1801BM1 can issue preliminary transaction address, so the reg.plm->reg.plr strobe
	// should be delayed till current transaction in progess completed
	//
	bTmpB1 = reg.qbus_aseq[1] || (wire.au_astb && reg.qbus_aseq[0]);
	BOOL qbus_nrdy = bTmpB1 && !reg.oe_clr;
	wire.uplr_stb = (bTmpB1 &&  reg.oe_clr) || (wire.au_astb && !reg.qbus_aseq[0]);
	//______________________________________________________________________________
	//
	// Control and glue logic
	//
	wire.sync_fedge = reg.sync_out && !reg.sync_ena;
	wire.din_start  = reg.oe_set || (reg.plrt[8] && wire.sync_fedge);
	wire.din_done   = reg.din_out_l && (reg.rply_ack[3] || reg.rply_ack[2]);

	bTmpB1 = !(reg.mjres || wire.mjres_rc || wire.din_done);
	wire.tplm_rc[1] = bTmpB1 && ((wire.uplr_stb && reg.tplmz[1]) || reg.tplm[1]); // instruction early prefetch
	wire.tplm_rc[2] = bTmpB1 && ((wire.uplr_stb && reg.tplmz[2]) || reg.tplm[2]); // data retrieving
	wire.tplm_rc[3] = bTmpB1 && ((wire.uplr_stb && reg.tplmz[3]) || reg.tplm[3]); // instruction fetch
	//
	// tplm1 - instruction prefetch
	// tplm3 - instruction fetch
	//
	wire.ir_seq_rc = !(reg.mjres || wire.mjres_rc) && (!reg.ir_seq || reg.tplm[1] || reg.tplm[3]) && (reg.plm23_wait || reg.ir_seq);
	wire.ir_clr = (!wire.tplm_rc[3] && reg.tplm[3]) || (!wire.ir_seq_rc && reg.ir_seq);

	wire.plm_ena_fc = !(reg.sop_out[0] || qbus_nrdy) && (reg.mjres || reg.ustb1_h || !reg.alu_busy_rp);
	wire.plm_stb_rc = !(reg.plm_stb || (reg.sop_out[3] || !(reg.sop_out[2] || reg.sop_out[1])));
	wire.ir_stb1	=   reg.tplm[1] || reg.tplm[3];

	//______________________________________________________________________________
	//
	// Qbus logic
	//
	wire.qbus_done = (wire.din_done && !reg.plrt[7]) || reg.dout_done || reg.mjres;
	wire.iako_out  = reg.din_out && reg.iak_flag;
	//
	// Master processor never asserts DMR_OUT
	//
	wire.dmr_out = reg.dmr_req && bPinPA_Not_Nul;
	//
	// Reply acknowlegement latches
	// (converted to flip-flops approach)
	//
	wire.rply_ack_fc = (wire.pin_rply_in && wire.pin_bsy_out);
	bTmpB1 = reg.plr[7] || reg.plr[8];
	wire.alu_qrdy = (!reg.qbus_adr || reg.plr[23] || !bTmpB1)	// wait for reg.areg free
				 && (!reg.dout_req || reg.plr[23] ||  bTmpB1)		// wait for write complete
				 && (reg.plr[10] || !(reg.tplm[2] || reg.iak_flag));				// wait for data or vector fetch
	wire.dmr_req_rc = (reg.sync_out || !reg.qbus_win) && (wire.au_astb || reg.dmr_req);
	wire.qbus_tena  = reg.dout_out || reg.din_out;
	wire.sync_stb   = reg.qbus_win && !(reg.sync_out_h && reg.qbus_win_h);

	wire.qbus_gnt  = (reg.qbus_nosr_h || (reg.sync_out_h && !reg.qbus_win_h)) && (reg.qbus_free_h || (wire.dmr_out && reg.sack_out));
	wire.qbus_nosr_rc = !(wire.pin_rply_in || wire.pin_sync_in);
	BOOL qbus_own = reg.qbus_own_fp || reg.qbus_own_rp;
	wire.qbus_flag_rc = !wire.qbus_done && (wire.sync_fedge || reg.qbus_flag);
	BOOL dmgo_out = (!(reg.iako_oe || qbus_own) && bPinPA_Not_Nul && wire.pin_dmr_in) || (!reg.qbus_req && reg.dmgi_in_l);
	BOOL dout_ext = reg.dout_out || reg.dout_out_l;
	wire.dout_req_rc = (!reg.plrt[8] && wire.au_qstbx && (wire.dmr_req_rc || reg.qbus_own_rp)) || (!wire.qbus_done && reg.dout_req);
	wire.dout_start = wire.dout_req_rc && wire.qbus_flag_rc && !reg.rply_ack[2]; // originally !reg.rply_ack[1]
	wire.oe_clr_fc  = reg.mjres || (!(wire.rply_ack_fc || reg.qbus_flag) && reg.rply_ack[1]);
	wire.oe_set_fc  = wire.qbus_gnt && reg.iak_flag && !(reg.dmr_req || qbus_own);

	//______________________________________________________________________________
	//
	//wire.sel177x = ((wire.pin_ad_in & 0177700) == 0177700) && (((wire.pin_ad_in >> 4) & 3) == wire.pin_pa);       
	BOOL sel_in  = reg.sel_xx && wire.pin_din_in;
	wire.sel_out = reg.sel_xx && wire.pin_dout_in;
	wire.rply_s1 = reg.sel_02 && wire.sel_out && reg.rply_s3;

	wire.au_qstbd = wire.iako_out || reg.tplm[2];

	//------------- ----------------
	//
	wire.tve_csr_wr = reg.sel_12 && wire.sel_out;
	wire.tve_lim_wr = reg.sel_06 && wire.sel_out;
	//
	// Hidden clock prescaler, generates the /4 and /128 frequencies
	//
	wire.tve_tclk128 = ((reg.tve_pre & 0177) == 0177);
	wire.pin_spclk = reg.tve_edge[1] && !reg.tve_edge[2];

	wire.tve_zero = (reg.tve_count == 0) && !reg.tve_csr[1];
	wire.tve_back = reg.tve_csr[4] && reg.tve_csr[1] && wire.pin_spclk && reg.tve_tclk4;
	wire.tve_load = wire.tve_zero && reg.tve_tclk4 && !reg.tve_csr[1];
	wire.tve_irq = reg.tve_intrq[0];

	//______________________________________________________________________________
	//
	// Interrupt controller
	//
	switch (m_nCpuType)
	{
	default:
	case CPU_VM1A:
		pli_matrix_a->eval(reg.rq, wire.pli);	//  
		break;
	case CPU_VM1G:
		pli_matrix_g->eval(reg.rq, wire.pli);	//  
		break;
	}
#ifdef DBG_OUT_PLI_LOG
	OutPLILog();
#endif

	//
	// Detector's reset
	//
	bTmpB1 = reg.plm1x_hl && !reg.plr[27];
	wire.aclo_ack = bTmpB1 && !reg.plir[3] && !reg.plir[2] &&  reg.plir[1];
	wire.uop_ack  = bTmpB1 && !reg.plir[3] && !reg.plir[2] && !reg.plir[1];
	wire.irq2_ack = bTmpB1 &&  reg.plir[3] &&  reg.plir[2] && !reg.plir[1];
	wire.irq3_ack = bTmpB1 &&  reg.plir[3] && !reg.plir[2] && !reg.plir[1];
	wire.tve_ack  = bTmpB1 &&  reg.plir[3] && !reg.plir[2] &&  reg.plir[1];
	wire.pli_req_rc = reg.plm23_ichk || reg.abort;
	wire.pli_stb = reg.pli_req;

	//______________________________________________________________________________
	//
	// Microcode state machine
	//
	WORD nMJ; //  .
	bita2data(reg.mj, nMJ, 15);
	switch (m_nCpuType)
	{
	default:
	case CPU_VM1A:
		plm_matrix_a->eval(reg.ir, nMJ, wire.plx);
		break;
	case CPU_VM1G:
		plm_matrix_g->eval(reg.ir, nMJ, wire.plx);
		break;
	}
#ifdef DBG_OUT_PLM_LOG
	OutPLMLog();
#endif

	bTmpB1 = wire.plm_ena_fc && !reg.plm[12];
	bTmpB2 = reg.plm[11] && reg.plm[13];
	wire.plm1x_fc = bTmpB1 &&  bTmpB2;
	wire.plm2x_fc = bTmpB1 && !bTmpB2;

	int nTmpInt1 = (reg.nPLM >> 1) & 7;							//  irq     check   wait IR     start IR        Note
	wire.plm23_fc[0] = wire.plm_ena_fc && (nTmpInt1 == 1);	//  no      yes     no          RTT opcode,     wait fetch completion
	wire.plm23_fc[1] = wire.plm_ena_fc && (nTmpInt1 == 3);	//  yes     yes     yes                         usual IR fetch
	wire.plm23_fc[2] = wire.plm_ena_fc && (nTmpInt1 == 5);	//  yes     no      no          WAIT opcode,    check irq only
	wire.plm23_fc[3] = wire.plm_ena_fc && (nTmpInt1 == 7);	//  yes     yes     no          !RTT opcode,    wait fetch completion
	wire.ir_set_fc = wire.plm23_fc[0] || wire.plm23_fc[1] || wire.plm23_fc[3];

	wire.plop = ((reg.nPLR >> 20) & 6) | (reg.plr[4] ? 1 : 0);
// 	wire.plop[0] = (nTmpInt1 == 0); // !reg.plr[22] && !reg.plr[21] && !reg.plr[4]; // 000
// 	wire.plop[1] = (nTmpInt1 == 1); // !reg.plr[22] && !reg.plr[21] &&  reg.plr[4]; // 001
// 	wire.plop[2] = (nTmpInt1 == 2); // !reg.plr[22] &&  reg.plr[21] && !reg.plr[4]; // 010
// 	wire.plop[3] = (nTmpInt1 == 3); // !reg.plr[22] &&  reg.plr[21] &&  reg.plr[4]; // 011
// 	wire.plop[4] = (nTmpInt1 == 4); //  reg.plr[22] && !reg.plr[21] && !reg.plr[4]; // 100
// 	wire.plop[5] = (nTmpInt1 == 5); //  reg.plr[22] && !reg.plr[21] &&  reg.plr[4]; // 101
// 	wire.plop[6] = (nTmpInt1 == 6); //  reg.plr[22] &&  reg.plr[21] && !reg.plr[4]; // 110
// 	wire.plop[7] = (nTmpInt1 == 7); //  reg.plr[22] &&  reg.plr[21] &&  reg.plr[4]; // 111

	//______________________________________________________________________________
	//
	// ALU
	//
	int nYAddr = ((reg.nPLR >> 25) & 0xf);
	int nXAddr = ((reg.nPLR >> 30) & 0xf);

	bTmpB1 = !(reg.plr[11] || reg.plr[13]);

	static WORD pRecode[16] = {
		-1, 7, 11, 3, 13, 5, 9, 1, -1, 6, 10, 2, 12, 4, 8, 0
	};
	WORD n;
	WORD rs0 = -1; //       ,
	if (reg.plr[13] && reg.plr[14])
	{
		rs0 = 12;	//,  rs0[x]    1   14 .
					//        rs0[12],    
					// .
	}
	else
	{
		n = pRecode[reg.yadr];
		if (bTmpB1)
		{
			rs0 = n;
		}
	}

	WORD rs1 = n = pRecode[reg.xadr];

	if (n == 12 || !reg.plr[20])
	{
		wire.rsw = 12;
	}
	else if (reg.plr[20])
	{
		wire.rsw = n;
	}
	else
	{
		wire.rsw = -1;
	}

	bTmpB2 = (reg.plr[13] && !reg.plr[14]) || reg.plr[11];
	BOOL bTmpB3 = (nYAddr == 2);
	BOOL vc_vsel =  bTmpB3 && bTmpB2;
	BOOL vc_csel = !bTmpB3 && bTmpB2;

	wire.xr = (rs1 < 14) ? reg.gpr[rs1] : 0;	//X bus
	wire.yr = (rs0 < 14) ? reg.gpr[rs0] : 0;	//Y bus
	wire.yr |= ((vc_vsel ? reg.vmux : 0) | (vc_csel ? reg.cmux : 0));
	/*
	// 
	wire.rs0[0]  = (reg.yadr == 15) && bTmpB1;
	wire.rs0[1]  = (reg.yadr ==  7) && bTmpB1;
	wire.rs0[2]  = (reg.yadr == 11) && bTmpB1;
	wire.rs0[3]  = (reg.yadr ==  3) && bTmpB1;
	wire.rs0[4]  = (reg.yadr == 13) && bTmpB1;
	wire.rs0[5]  = (reg.yadr ==  5) && bTmpB1;
	wire.rs0[6]  = (reg.yadr ==  9) && bTmpB1;
	wire.rs0[7]  = (reg.yadr ==  1) && bTmpB1;
	wire.rs0[8]  = (reg.yadr == 14) && bTmpB1;
	wire.rs0[9]  = (reg.yadr ==  6) && bTmpB1;
	wire.rs0[10] = (reg.yadr == 10) && bTmpB1;
	wire.rs0[11] = (reg.yadr ==  2) && bTmpB1;
	wire.rs0[12] = (reg.yadr == 12) && bTmpB1 || (reg.plr[13] && reg.plr[14]);
	wire.rs0[13] = (reg.yadr ==  4) && bTmpB1;

	wire.rs1[0]  = (reg.xadr == 15);
	wire.rs1[1]  = (reg.xadr ==  7);
	wire.rs1[2]  = (reg.xadr == 11);
	wire.rs1[3]  = (reg.xadr ==  3);
	wire.rs1[4]  = (reg.xadr == 13);
	wire.rs1[5]  = (reg.xadr ==  5);
	wire.rs1[6]  = (reg.xadr ==  9);
	wire.rs1[7]  = (reg.xadr ==  1);
	wire.rs1[8]  = (reg.xadr == 14);
	wire.rs1[9]  = (reg.xadr ==  6);
	wire.rs1[10] = (reg.xadr == 10);
	wire.rs1[11] = (reg.xadr ==  2);
	wire.rs1[12] = (reg.xadr == 12);
	wire.rs1[13] = (reg.xadr ==  4);

	wire.rsw[0]  = (reg.xadr == 15) && reg.plr[20];
	wire.rsw[1]  = (reg.xadr ==  7) && reg.plr[20];
	wire.rsw[2]  = (reg.xadr == 11) && reg.plr[20];
	wire.rsw[3]  = (reg.xadr ==  3) && reg.plr[20];
	wire.rsw[4]  = (reg.xadr == 13) && reg.plr[20];
	wire.rsw[5]  = (reg.xadr ==  5) && reg.plr[20];
	wire.rsw[6]  = (reg.xadr ==  9) && reg.plr[20];
	wire.rsw[7]  = (reg.xadr ==  1) && reg.plr[20];
	wire.rsw[8]  = (reg.xadr == 14) && reg.plr[20];
	wire.rsw[9]  = (reg.xadr ==  6) && reg.plr[20];
	wire.rsw[10] = (reg.xadr == 10) && reg.plr[20];
	wire.rsw[11] = (reg.xadr ==  2) && reg.plr[20];
	wire.rsw[12] = (reg.xadr == 12) || !reg.plr[20];
	wire.rsw[13] = (reg.xadr ==  4) && reg.plr[20];

	wire.xr = (wire.rs1[0]  ? reg.gpr[0]  : 0)
			| (wire.rs1[1]  ? reg.gpr[1]  : 0)
			| (wire.rs1[2]  ? reg.gpr[2]  : 0)
			| (wire.rs1[3]  ? reg.gpr[3]  : 0)
			| (wire.rs1[4]  ? reg.gpr[4]  : 0)
			| (wire.rs1[5]  ? reg.gpr[5]  : 0)
			| (wire.rs1[6]  ? reg.gpr[6]  : 0)
			| (wire.rs1[7]  ? reg.gpr[7]  : 0)
			| (wire.rs1[8]  ? reg.gpr[8]  : 0)
			| (wire.rs1[9]  ? reg.gpr[9]  : 0)
			| (wire.rs1[10] ? reg.gpr[10] : 0)
			| (wire.rs1[11] ? reg.gpr[11] : 0)
			| (wire.rs1[12] ? reg.gpr[12] : 0)
			| (wire.rs1[13] ? reg.gpr[13] : 0);

	wire.yr = (wire.rs0[0]  ? reg.gpr[0] : 0)
			| (wire.rs0[1]  ? reg.gpr[1] : 0)
			| (wire.rs0[2]  ? reg.gpr[2] : 0)
			| (wire.rs0[3]  ? reg.gpr[3] : 0)
			| (wire.rs0[4]  ? reg.gpr[4] : 0)
			| (wire.rs0[5]  ? reg.gpr[5] : 0)
			| (wire.rs0[6]  ? reg.gpr[6] : 0)
			| (wire.rs0[7]  ? reg.gpr[7] : 0)
			| (wire.rs0[8]  ? reg.gpr[8] : 0)
			| (wire.rs0[9]  ? reg.gpr[9] : 0)
			| (wire.rs0[10] ? reg.gpr[10] : 0)
			| (wire.rs0[11] ? reg.gpr[11] : 0)
			| (wire.rs0[12] ? reg.gpr[12] : 0)
			| (wire.rs0[13] ? reg.gpr[13] : 0)
			| (wire.vc_vsel ? reg.vmux : 0)
			| (wire.vc_csel ? reg.cmux : 0);

	ASSERT(nxr == wire.xr);
	ASSERT(nyr == wire.yr);
*/
	wire.au_pswy = (nYAddr == 8) && bTmpB1;
	wire.au_qsy  = (nYAddr == 0) && bTmpB1;
	wire.au_pswx = (nXAddr == 8);
	wire.au_qsx  = (nXAddr == 0);

	wire.au_is0 = !(reg.plr[13] && reg.plr[14] && reg.plr[25] && reg.plr[26] && !reg.plr[27]);

	wire.au_pstbx = wire.au_pswx && reg.plr[20];


	//______________________________________________________________________________
	//
	// ALU function unit
	//
	WORD nx = reg.alu_x ? ~reg.xreg : reg.xreg;
	//
	// ALU controls
	//
	//      plr[17:13]  cl      alu_b   alu_c   alu_d   alu_e   alu_x  oxy    axy     h       alu
	//      00x00       1       0       1       0       0       0     !x||!y !x&&!y  !(x^y)  x+y
	//      00x01       1       0       1       0       0       0       -     -       -      -
	//      00x11       1       0       1       0       0       0       -     -       -      -
	//      00x10       1       0       0       0       1       0     !x||!y  !x     !(x&&!y)  x&&!y
	//      01x00       0       0       1       0       0       1      x||!y  x&&!y    x^y        y-x
	//      01x01       0       0       1       0       0       1       -     -       -      -
	//      01x11       0       0       1       0       0       1       -     -       -      -
	//      01x10       0       0       1       0       1       0     !x||!y !x&&!y   !(x^y)  x^y
	//      10x00       1       1       1       0       1       0     !x||y  !x&&!y    !y      y
	//      10x01       1       0       0       0       1       0       1    !x       !x         x
	//      10x10       1       0       1       0       1       0       1   !x&&!y  !(x||y)   x||y
	//      10x11       1       0       0       0       1       0       1    !x        !x         x
	//      11x00       0       1       0       1       0       0     !x||y  !x&&y      x^y       x-y
	//      11x01       0       1       0       1       0       0        -       -          -        -
	//      11x11       0       1       0       1       0       0        -       -          -        -
	//      11x10       1       1       0       0       1       0     !x||y  !x      !(x&&y)  x&&y
	//
	bTmpB1 = !wire.alu_e_fc && reg.plr[16];
	wire.alu_e_fc = (!reg.plr[13] && reg.plr[14]) || (!reg.plr[16] && reg.plr[17]);
	wire.alu_b_fc =   reg.plr[17] && (reg.plr[16] || !(reg.plr[13] || reg.plr[14]));
	wire.alu_c_fc = (!reg.plr[13] && (reg.plr[16] ^ reg.plr[17])) || !(wire.alu_e_fc || reg.plr[17]);
	wire.alu_d_fc = bTmpB1 &&  reg.plr[17];
	wire.alu_x_fc = bTmpB1 && !reg.plr[17];
	wire.alu_s_fc = !(reg.plr[13] && reg.plr[14]) || (reg.plr[25] && reg.plr[26]);
	wire.cl_fc    = !reg.plr[16] || (reg.plr[17] && wire.alu_e_fc);

	//
	// ALU and/or products
	//
	WORD oxy = ~(nx & ((reg.plr[17] ? 0 : ~reg.yreg) | (reg.alu_b ? reg.yreg : 0)));
	WORD axy = ~(nx | (reg.alu_d ? reg.yreg : 0) | (reg.alu_c ? ~reg.yreg : 0));
	//
	// ALU selectable function
	//
	WORD h = ~(oxy | axy) | (oxy & axy);
	BOOL bAXY[16], bOXY[16], bC[16], bF[16], bH[16];
	data2bita(oxy, bOXY, 16);
	data2bita(axy, bAXY, 16);
	data2bita(h, bH, 16);
	BOOL cn, cl = reg.cl;
	for (int i = 0; i < 16; ++i)
	{
		bC[i] = cn = (cl || bAXY[i]) && bOXY[i];	//carry
		bF[i] = reg.alu_e ? bH[i] : !(cl ^ bH[i]);		//ALU function inverted output
		cl = cn;
	}
	WORD f;
	bita2data(bF, f, 16);
	//
	// ALU result shifter
	//
	BOOL fctl[8];
	fctl[2] = !(reg.plr[25] || reg.plr[26]) && reg.plr[27];
	fctl[1] = fctl[2] && !reg.plr[18];
	fctl[3] = reg.plr[27] && reg.plr[18];
	bTmpB1 = reg.plr[13] && reg.plr[14] && reg.plr[27];
	fctl[4] = bTmpB1 && !reg.plr[25] &&  reg.plr[26];
	fctl[6] = bTmpB1 &&  reg.plr[25] && !reg.plr[26];
	fctl[5] = fctl[4] && !reg.plr[18];
	fctl[7] = fctl[6] && !reg.plr[18];

	BOOL fbitc = !(reg.psw[0] && !reg.plr[25] && reg.plr[26]);
	BOOL fbit7 = fctl[1] && bF[7]
			  || fctl[3] && bF[8]
			  || fctl[7] && bC[7]
			  || fctl[5] && fbitc;
	BOOL fbit15 = fctl[2] && bF[15]
			   || fctl[6] && bC[15]
			   || fctl[4] && fbitc;

	//
	// alu_shr = !alu_s &&  reg.plr[27];
	// alu_shl = !alu_s && !reg.plr[27];
	//
	WORD alu;
	if (reg.alu_s)
	{
		alu = ~f;
	}
	else if (reg.plr[27])
	{
		alu = ~((fbit15 ? 0100000 : 0) | ((f & 0177000) >> 1)
			  | (fbit7  ? 0200    : 0) | ((f & 0376)    >> 1));
	}
	else
	{
		alu = ~(((f & 077777) << 1) | (fbitc ? 1 : 0));
	};

	//
	// X bus (12 entries)
	//    AU_RSX0  - general purpose regs
	//    AU_RSX1  - general purpose regs
	//    AU_QSX   - qbus temporary reg
	//    AU_PSWX  - PSW
	//
	//    AU_ALSx  - ALU result strobe     (writeonly)
	//    AU_ASTB  - A address register    (readonly)
	//    AU_ASTB  - A address register    (readonly)
	//    AU_QSTBX - Qbus temporary reg    (readonly)
	//    AU_QSTBX - Qbus temporary reg    (readonly)
	//    AU_IS0   - ALU X argument        (readonly)
	//    AU_IS1   - ALU X argument        (readonly)
	//
	// Y bus, inverted (9 entries)
	//    AU_RSY0  - general purpose regs
	//    AU_RSY1  - general purpose regs
	//    AU_QSY   - Qbus temporary reg
	//    AU_PSWY  - PSW
	//
	//    AU_ALSx  - ALU result strobe     (writeonly))
	//    AU_VSELY - vector generator      (Y-writeonly)
	//    AU_CSELY - constant generator    (Y-writeonly)
	//    ALU_U    - ALU Y argument        (readonly)
	//
	WORD nPSW;
	bita2data(reg.psw, nPSW, 16);
	wire.x = reg.au_alsl ? (reg.au_alsh ? alu : alu & 0377) :
		(wire.xr
		| (wire.au_qsx ? reg.qreg : 0)
		| (wire.au_pswx ? nPSW : 0));

	wire.y = ~(wire.yr
			| (wire.au_qsy ? reg.qreg : 0)
			| (wire.au_pswy ? nPSW : 0));

	//
	// ALU result condition flags
	//
	BOOL fmux[2];
	fmux[0] = reg.cl ? (reg.alu_e || (reg.plr[18] ? bC[15] : bC[7])) : !(reg.alu_e || (reg.plr[18] ? bC[15] : bC[7]));
	fmux[1] = (bF[0] || !reg.plr[27]) && (reg.plr[27] || (reg.plr[18] ? bF[15] : bF[7]));

	wire.flag[0] = reg.alu_s ? !fmux[0] : !fmux[1];									// C
	wire.flag[2] = (!reg.plr[18] || ((alu & 0177400) == 0)) && ((alu & 0377) == 0);	// Z
	wire.flag[3] = reg.plr[18] ? GETBIT(alu, 15) : GETBIT(alu, 7);					// N
	wire.flag[1] = !(reg.alu_s && reg.alu_e)										// V
		&& !(((fmux[1] ^ wire.flag[3]) || reg.alu_s) && (reg.plr[18] ? (bC[14] ^ !bC[15]) : (bC[6] ^ !bC[7])));

	if (sel_in)
	{
		if (reg.sel_00)
		{
			WORD n;
			bita2data(reg.reg_csr, n, 3);
			wire.pin_ad_out = 0177740 | ((wire.pin_pa & 3) << 3) | (n & 7);
		}
		else if (reg.sel_02)
		{
			wire.pin_ad_out = 0177777;
		}
		else if (reg.sel_04)
		{
			WORD n;
			bita2data(reg.reg_err, n, 8);
			wire.pin_ad_out = n & 0377;
		}
		//
		// Reading the timer registers content to the shared bus
		//
		else if (reg.sel_06)
		{
			wire.pin_ad_out = reg.tve_limit;
		}
		else if (reg.sel_10)
		{
			wire.pin_ad_out = reg.tve_count;
		}
		if (reg.sel_12)
		{
			WORD n = 0;
			bita2data(reg.tve_csr, n, 8);
			wire.pin_ad_out = 0177400 | (n & 0377);
		}
	}
	else if (wire.sync_stb)
	{
		wire.pin_ad_out = reg.areg;
	}
	else if (dout_ext)
	{
		wire.pin_ad_out = reg.au_ta0 ? ((reg.qreg & 0377) << 8) : (reg.plrt[6] ? (reg.qreg & 0377) : reg.qreg);
	}
	else
	{
		wire.pin_ad_out = 0;
	}
	wire.pin_ad_ena = wire.sync_stb || dout_ext || !(reg.sel_16 || reg.sel_14 || !sel_in);

	//
	// External connection assignments
	//

	wire.pin_dout_out = reg.dout_out && reg.dout_out_l;
	wire.pin_din_out  = reg.din_out;
	wire.pin_wtbt_out = (wire.sync_stb && !reg.plrt[8]) || (dout_ext && reg.plrt[6]);
	wire.pin_sync_out = reg.sync_out;
	wire.pin_sync_ena = reg.sync_ena || reg.sync_out;
	wire.pin_rply_out = (sel_in || wire.sel_out) && (wire.pin_rply_in || !(reg.sel_02 && reg.rply_s2));
	wire.pin_iako_out = reg.iako_out_lh;
	wire.pin_dmr_out  = wire.dmr_out;
	wire.pin_sack_out = reg.sack_out;
	wire.pin_dmgo_out = dmgo_out;

	wire.pin_sel2_out = reg.sel_14;
	wire.pin_sel1_out = reg.sel_16;


//----------------------------------------------------------------------------
	//
	// Shared Qbus lines
	//
	if (wire.pin_ad_ena)
	{
		m_IOPins.pin_ad_n = ~(wire.pin_ad_out);
	}

	m_IOPins.pin_dout_n = wire.pin_ctrl_ena ? !wire.pin_dout_out : TRUE;
	m_IOPins.pin_din_n  = wire.pin_ctrl_ena ? !wire.pin_din_out  : TRUE;
	m_IOPins.pin_wtbt_n = wire.pin_ctrl_ena ? !wire.pin_wtbt_out : TRUE;
	m_IOPins.pin_iako_n = wire.pin_ctrl_ena ? !wire.pin_iako_out : TRUE;

	m_IOPins.pin_sync_n = wire.pin_sync_ena ? !wire.pin_sync_out : TRUE;

	m_IOPins.pin_dmgo_n = !wire.pin_dmgo_out;
	//
	// "Open collector" outputs
	//
	m_IOPins.pin_init_n = wire.pin_init_out ? FALSE : TRUE;
	m_IOPins.pin_sack_n = wire.pin_sack_out ? FALSE : TRUE;
	m_IOPins.pin_rply_n = wire.pin_rply_out ? FALSE : TRUE;
	m_IOPins.pin_dmr_n  = wire.pin_dmr_out  ? FALSE : TRUE;
	m_IOPins.pin_bsy_n  = wire.pin_bsy_out  ? FALSE : TRUE;
	m_IOPins.pin_sel1_n = wire.pin_sel1_out ? FALSE : TRUE;
	m_IOPins.pin_sel2_n = wire.pin_sel2_out ? FALSE : TRUE;

}

void C1801VM1::always_posedge_pin_clk_p_or_posedge_tve_reset() // 
{
	if (wire.pin_init_out || wire.pin_init_in)
	{
		//
		// Asynchronous prescaler reset, no in original 1801VM1,
		// added here to provide the comfortable simulation
		//
		reg.tve_pre = 0;
		reg.tve_div = 0;
		reg.tve_tclk4 = 0;
		reg.tve_tclk = 0;
		for (int i = 0; i < 3; ++i)
		{
			reg.tve_edge[i] = FALSE;
		}
		reg.tve_intrq[0] = FALSE;
		reg.tve_intrq[1] = FALSE;
		//
		// Timer counter, preload/capture and control registers
		//
		//
		// Asynchronous timer counter and preload reset, no in original 1801VM1,
		// added here to provide the comfortable simulation. The control timer
		// register reset is implemented in the original chip
		//
		for (int i = 0; i < 8; ++i)
		{
			reg.tve_csr[i] = FALSE;
		}
		reg.tve_count = 0;
		reg.tve_limit = 0;
	}
	else
	{
		//
		// Timer counter, preload/capture and control registers
		//
		//
		// Timer control register
		//
		// csr[0]   - selects external clock source
		// csr[1]   - selects capture mode
		// csr[2]   - enables interrupt request
		// csr[3]   - selects one-shot mode
		// csr[4]   - timer count enable
		// csr[5]   - 1/16 divider enable
		// csr[6]   - 1/4 divider enable
		// csr[7]   - interrupt overflow flag
		//
		if (wire.tve_csr_wr)
		{
			//
			// The prioritized CSR write from bus
			// Clocked every system bus cycle
			//
			for (int i = 0; i < 8; ++i)
			{
				reg.tve_csr[i] = wire.wb_pio_dat_i[i];
			}
			//
			// Counter is written unconditionally at every CSR bus write
			//
			reg.tve_count = snap_reg.tve_limit;
		}

		if (wire.tve_lim_wr)
		{
			//
			// The prioritized register write from bus
			// Clocked every system bus cycle
			//
			//WORD n;
			//bita2data(wire.wb_pio_dat_i, n, 16);
			reg.tve_limit = wire.pin_ad_in; //n;
		}

		if (m_IOPins.pin_ena)
		{
			reg.tve_pre = (snap_reg.tve_pre + 1) & 0177;
			reg.tve_tclk4 = (snap_reg.tve_pre & 3) == 3;
			//
			// Documented divisor 1/4, 1/16, 1/64
			//
			if (wire.tve_tclk128)
			{
				reg.tve_div = (snap_reg.tve_div + 1) & 077;
			}

			//
			// Timer clock selector
			//
			/*
			reg.tve_tclk = !snap_reg.tve_csr[0] && !snap_reg.tve_csr[6] && !snap_reg.tve_csr[5] && wire.tve_tclk128
						|| !snap_reg.tve_csr[0] &&  snap_reg.tve_csr[6] && !snap_reg.tve_csr[5] && wire.tve_tclk128 && ((snap_reg.tve_div & 3) == 3)
						|| !snap_reg.tve_csr[0] && !snap_reg.tve_csr[6] &&  snap_reg.tve_csr[5] && wire.tve_tclk128 && ((snap_reg.tve_div & 017) == 017)
						|| !snap_reg.tve_csr[0] &&  snap_reg.tve_csr[6] &&  snap_reg.tve_csr[5] && wire.tve_tclk128 && ((snap_reg.tve_div & 077) == 077)
						||  snap_reg.tve_csr[0] && wire.pin_spclk && ((snap_reg.tve_pre & 3) == 3);
			*/
			reg.tve_tclk = !snap_reg.tve_csr[0] && wire.tve_tclk128 &&
						  (!snap_reg.tve_csr[6] && !snap_reg.tve_csr[5]
						||  snap_reg.tve_csr[6] && !snap_reg.tve_csr[5] && ((snap_reg.tve_div & 3) == 3)
						|| !snap_reg.tve_csr[6] &&  snap_reg.tve_csr[5] && ((snap_reg.tve_div & 017) == 017)
						||  snap_reg.tve_csr[6] &&  snap_reg.tve_csr[5] && ((snap_reg.tve_div & 077) == 077))
						||  snap_reg.tve_csr[0] && wire.pin_spclk && ((snap_reg.tve_pre & 3) == 3);

			//
			// Falling edge detector (raising for the inverted input pin_sp)
			//
			if (snap_reg.tve_tclk4)
			{
				//
				// Input metastability eliminator and synchronizer
				//
				reg.tve_edge[0] = wire.pin_sp;
				reg.tve_edge[1] = snap_reg.tve_edge[0];
				reg.tve_edge[2] = snap_reg.tve_edge[1];
			}
			//
			// Timer counter, preload/capture and control registers
			//
			if (!wire.tve_csr_wr)
			{
				if (wire.tve_zero && snap_reg.tve_csr[3])
				{
					//
					// Reset the RUN bit in one-shot mode
					//
					reg.tve_csr[4] = FALSE;
				}

				if (snap_reg.tve_intrq[1] && snap_reg.tve_csr[2] && (wire.tve_zero || wire.tve_back))
				{
					//
					// Set the interrupt overflow flag if enabled
					//
					reg.tve_csr[7] = TRUE;
				}
			}
			//
			// Timer preload and capture value register
			//
			if (!wire.tve_lim_wr && wire.tve_back)
			{
				//
				// Capture the current counter value
				//
				reg.tve_limit = snap_reg.tve_count;
			}
			//
			// Decrementing counter register
			//
			if (!wire.tve_csr_wr)
			{
				if (wire.tve_load)
				{
					//
					// Counter is being reloaded
					//
					reg.tve_count = snap_reg.tve_limit;
				}
				else if (snap_reg.tve_tclk && snap_reg.tve_csr[4])
				{
					//
					// Count selected clock events if enabled
					//
					reg.tve_count = snap_reg.tve_count - 1;
				}
			}
		}

		if (wire.tve_ack)
		{
			//
			// New timer interrupt request raising while 
			// acknowledgement can be missed
			//
			reg.tve_intrq[0] = FALSE;
		}
		else if (m_IOPins.pin_ena)
		{
			//
			// Interrupt flags
			//
			if (snap_reg.tve_csr[2] && (wire.tve_zero || wire.tve_back))
			{
				reg.tve_intrq[0] = TRUE;
			}
		}

		if (!(snap_reg.tve_csr[2] && (wire.tve_zero || wire.tve_back)))
		{
			reg.tve_intrq[1] = snap_reg.tve_intrq[0];
		}
	}
}


void C1801VM1::always_posedge_pin_clk_n()
{
	//______________________________________________________________________________
	//
	// Control and glue logic
	//
	reg.plm_ena = wire.plm_ena_fc;

	reg.sop_out[1] = snap_reg.sop_out[0];
	reg.sop_out[3] = snap_reg.ir_stop || wire.pli_stb || snap_reg.ir_stb2 || snap_reg.mjres_h || (snap_reg.alu_nrdy && ((wire.plop == 0) || (wire.plop == 4)));

	reg.ustb1 = snap_reg.ustb_h;
	reg.ustb1_hl = snap_reg.ustb1_h;

	reg.ir_set = wire.ir_set_fc;

	//______________________________________________________________________________
	//
	// Qbus logic
	//
	reg.rply_ack[2] = snap_reg.rply_ack[1];
	reg.dmr_req_l = snap_reg.dmr_req;

	reg.dout_done = snap_reg.rply_ack[1] && snap_reg.dout_out;
	reg.dout_out_l = snap_reg.dout_out;
	reg.din_out_l = snap_reg.din_out;

	reg.oe_clr = wire.oe_clr_fc;
	reg.oe_set = wire.oe_set_fc;

	reg.sync_out = snap_reg.qbus_win_h;
	reg.sync_ena = snap_reg.sync_out_h;
	reg.qbus_gnt_l = wire.qbus_gnt;

	reg.dmgi_in_l = wire.pin_dmgi;
	reg.rply_s3 = !snap_reg.rply_s2;
	reg.rply_s2 = snap_reg.start_irq;

	if (wire.oe_clr_fc)
	{
		reg.iako_oe = FALSE;
	}
	else if (wire.oe_set_fc)
	{
		reg.iako_oe = TRUE;
	}

	if (wire.oe_clr_fc)
	{
		reg.qbus_win = FALSE;
	}
	else if (snap_reg.dmr_req && wire.qbus_gnt)
	{
		reg.qbus_win = TRUE;
	}

	if (snap_reg.qbus_own_rp)
	{
		reg.qbus_own_fp = FALSE;
	}
	else if (snap_reg.dmr_req && wire.qbus_gnt)
	{
		reg.qbus_own_fp = TRUE;
	}


	//______________________________________________________________________________
	//
	// Microcode state machine
	//
	reg.plm23_wait = wire.plm23_fc[0] || wire.plm23_fc[3];
	reg.plm23_ichk = wire.plm23_fc[1] || wire.plm23_fc[2] || wire.plm23_fc[3];

	reg.plm1x = wire.plm1x_fc;
	reg.plm2x = wire.plm2x_fc;
	reg.plm1x_hl = snap_reg.plm1x;

	if (snap_reg.plm_ena)
	{
		reg.psw_stb  = !((wire.plop == 0) || (wire.plop == 2) || (wire.plop == 7));
		//reg.psw_stbc = !wire.plop[0] && !wire.plop[2] && !wire.plop[7] && !wire.plop[1] && !wire.plop[3];
		reg.psw_stbc = reg.psw_stb && !((wire.plop == 1) || (wire.plop == 3));
		BOOL btmp1 = !((wire.plop == 1) || (wire.plop == 5) || (wire.plop == 7));
		reg.psw_mj = (wire.plop == 7) && snap_reg.ir_stb2;
		//reg.mj_stb1 = (wire.plop[7] && snap_reg.ir_stb2) || btmp1;
		//reg.mj_stb2 = (wire.plop[7] && snap_reg.ir_stb2) || (btmp1 && !wire.plop[3]);
		reg.mj_stb1 = (reg.psw_mj) || btmp1;
		reg.mj_stb2 = (reg.psw_mj) || (btmp1 && (wire.plop != 3));
	}
	//______________________________________________________________________________
	//
	if (wire.plm_ena_fc)
	{
		for (int i = 0; i < 34; ++i)
		{
			reg.plr[i] = snap_reg.plm[i];
		}
		bita2data(reg.plr, reg.nPLR, 34);
	}
	//
	// ALU result flags and PSW
	//
	if (snap_reg.au_alsl)
	{
		reg.freg[0] = wire.flag[0];
		reg.freg[1] = wire.flag[1];
		reg.freg[2] = wire.flag[2];
		reg.freg[3] = wire.flag[3];
	}

	//
	// ALU general purpose registers
	//

	reg.xadr = (snap_reg.nPLR >> 30) & 0xf;
	reg.yadr = (snap_reg.nPLR >> 25) & 0xf;

	reg.vc_vreg[0] = snap_reg.vsel[0];
	reg.vc_vreg[1] = snap_reg.vsel[1];
	reg.vc_vreg[2] = snap_reg.vsel[2];
	reg.vc_vreg[3] = snap_reg.vsel[3];

	if (wire.rsw < 14)
	{
		if (snap_reg.au_alsl)
		{
			//  
			reg.gpr[wire.rsw] &= 0xff00;
			reg.gpr[wire.rsw] |= wire.x & 0xff;
		}
		if (snap_reg.au_alsh)
		{
			//  
			reg.gpr[wire.rsw] &= 0xff;
			reg.gpr[wire.rsw] |= wire.x & 0xff00;
		}
	}

	/*
	// 
	if (snap_reg.au_alsl)
	{
		if (wire.rsw[0])  reg.gpr[0] = (reg.gpr[0] & 0177400) | (wire.x & 0377);
		if (wire.rsw[1])  reg.gpr[1] = (reg.gpr[1] & 0177400) | (wire.x & 0377);
		if (wire.rsw[2])  reg.gpr[2] = (reg.gpr[2] & 0177400) | (wire.x & 0377);
		if (wire.rsw[3])  reg.gpr[3] = (reg.gpr[3] & 0177400) | (wire.x & 0377);
		if (wire.rsw[4])  reg.gpr[4] = (reg.gpr[4] & 0177400) | (wire.x & 0377);
		if (wire.rsw[5])  reg.gpr[5] = (reg.gpr[5] & 0177400) | (wire.x & 0377);
		if (wire.rsw[6])  reg.gpr[6] = (reg.gpr[6] & 0177400) | (wire.x & 0377);
		if (wire.rsw[7])  reg.gpr[7] = (reg.gpr[7] & 0177400) | (wire.x & 0377);
		if (wire.rsw[8])  reg.gpr[8] = (reg.gpr[8] & 0177400) | (wire.x & 0377);
		if (wire.rsw[9])  reg.gpr[9] = (reg.gpr[9] & 0177400) | (wire.x & 0377);
		if (wire.rsw[10]) reg.gpr[10] = (reg.gpr[10] & 0177400) | (wire.x & 0377);
		if (wire.rsw[11]) reg.gpr[11] = (reg.gpr[11] & 0177400) | (wire.x & 0377);
		if (wire.rsw[12]) reg.gpr[12] = (reg.gpr[12] & 0177400) | (wire.x & 0377);
		if (wire.rsw[13]) reg.gpr[13] = (reg.gpr[13] & 0177400) | (wire.x & 0377);
	}

	if (snap_reg.au_alsh)
	{
		if (wire.rsw[0])  reg.gpr[0] = (reg.gpr[0] & 0377) | (wire.x & 0177400);
		if (wire.rsw[1])  reg.gpr[1] = (reg.gpr[1] & 0377) | (wire.x & 0177400);
		if (wire.rsw[2])  reg.gpr[2] = (reg.gpr[2] & 0377) | (wire.x & 0177400);
		if (wire.rsw[3])  reg.gpr[3] = (reg.gpr[3] & 0377) | (wire.x & 0177400);
		if (wire.rsw[4])  reg.gpr[4] = (reg.gpr[4] & 0377) | (wire.x & 0177400);
		if (wire.rsw[5])  reg.gpr[5] = (reg.gpr[5] & 0377) | (wire.x & 0177400);
		if (wire.rsw[6])  reg.gpr[6] = (reg.gpr[6] & 0377) | (wire.x & 0177400);
		if (wire.rsw[7])  reg.gpr[7] = (reg.gpr[7] & 0377) | (wire.x & 0177400);
		if (wire.rsw[8])  reg.gpr[8] = (reg.gpr[8] & 0377) | (wire.x & 0177400);
		if (wire.rsw[9])  reg.gpr[9] = (reg.gpr[9] & 0377) | (wire.x & 0177400);
		if (wire.rsw[10]) reg.gpr[10] = (reg.gpr[10] & 0377) | (wire.x & 0177400);
		if (wire.rsw[11]) reg.gpr[11] = (reg.gpr[11] & 0377) | (wire.x & 0177400);
		if (wire.rsw[12]) reg.gpr[12] = (reg.gpr[12] & 0377) | (wire.x & 0177400);
		if (wire.rsw[13]) reg.gpr[13] = (reg.gpr[13] & 0377) | (wire.x & 0177400);
	}
	*/
	reg.au_alsl = snap_reg.alu_busy_rp && snap_reg.ustb_h;
	//reg.au_alsh = snap_reg.alu_busy_rp && snap_reg.ustb_h && snap_reg.plr[18];
	reg.au_alsh = reg.au_alsl && snap_reg.plr[18];
	BOOL btmp2 = snap_reg.plr[7] || snap_reg.plr[8];
	BOOL btmp1 = snap_reg.alu_busy_rp && snap_reg.ustb_h && !((!snap_reg.plr[14] && snap_reg.plr[13]) || snap_reg.plr[23]);
	reg.au_astb_xa = btmp1 &&  btmp2;
	reg.au_qstb_xa = btmp1 && !btmp2;
	btmp1 = !(snap_reg.plr[14] || snap_reg.plr[23]) && snap_reg.plr[13];
	reg.au_astb_xu = btmp1 &&  btmp2;
	reg.au_qstb_xu = btmp1 && !btmp2;
	//
	// ALU qbus register
	//
	if (wire.au_qstbx)
	{
		reg.qreg = wire.x;
	}
	else if (wire.au_qstbd)
	{
		//reg.qreg = (reg.qreg & 0177400) | ((snap_reg.au_ta0) ? ((wire.pin_ad_in & 0177400) >> 8) : (wire.pin_ad_in & 0377)); //  reg.qreg  
		//reg.qreg = (reg.qreg & 0377) | (wire.pin_ad_in & 0177400); //  reg.qreg  
		reg.qreg = (wire.pin_ad_in & 0177400) | ((snap_reg.au_ta0) ? ((wire.pin_ad_in & 0177400) >> 8) : (wire.pin_ad_in & 0377));
	}

	if (wire.au_astb)
	{
		reg.areg = wire.x;
	}
	//
	// ALU input parameters (from X and Y buses) registers
	//
	reg.xreg = (wire.au_is0) ? wire.x : ((wire.x & 0xff) << 8) | ((wire.x & 0xff00) >> 8);

	reg.yreg = wire.y;

	reg.cl    = wire.cl_fc;
	reg.alu_b = wire.alu_b_fc;
	reg.alu_c = wire.alu_c_fc;
	reg.alu_d = wire.alu_d_fc;
	reg.alu_e = wire.alu_e_fc;
	reg.alu_x = wire.alu_x_fc;
	reg.alu_s = wire.alu_s_fc;
}


void C1801VM1::always_posedge_pin_clk_p()
{
	//______________________________________________________________________________
	//
	// Control and glue logic
	//
	reg.ir_seq = wire.ir_seq_rc;
	reg.sop_out[2] = snap_reg.sop_out[1];

	if (wire.plm_stb_rc)
	{
		reg.sop_out[0] = FALSE;
	}
	else if (wire.mjres_rc || snap_reg.plm_ena)
	{
		reg.sop_out[0] = TRUE;
	}

	reg.plm_stb = wire.plm_stb_rc;
	reg.ustb1_h = snap_reg.ustb1;
	reg.tplm[0] = wire.tplm_rc[0];
	reg.tplm[1] = wire.tplm_rc[1];
	reg.tplm[2] = wire.tplm_rc[2];
	reg.tplm[3] = wire.tplm_rc[3];

	//
	//      plr[6]  QBUS operation type: byte operation flag
	//      plr[7]  QBUS operation type: write flag
	//      plr[8]  QBUS operation type: read flag
	//                      00x - nothing
	//                      010 - write word
	//                      011 - write byte
	//                      100 - read word
	//                      101 - read byte
	//                      110 - read-modify-write word
	//                      111 - read-modify-write byte
	//
	reg.plrtz[6] = snap_reg.plr[6];
	reg.plrtz[7] = snap_reg.plr[7] && snap_reg.plr[8];
	reg.plrtz[8] = snap_reg.plr[8];

	//
	// Reset Read Flag after read completion within
	// read-modify-write operations
	//
	if (wire.din_done)
	{
		reg.plrt[8] = FALSE;
	}
	else if (wire.uplr_stb)
	{
		reg.plrt[8] = snap_reg.plrtz[8];
	}

	if (wire.uplr_stb)
	{
		reg.plrt[7] = snap_reg.plrtz[7];
		reg.plrt[6] = snap_reg.plrtz[6];
	}

	//______________________________________________________________________________
	//
	// Qbus logic
	//
	reg.iako_out_lh = wire.iako_out;
	reg.rply_ack[1] = wire.rply_ack_fc;
	reg.rply_ack[3] = snap_reg.rply_ack[2];

	if (wire.uplr_stb)
	{
		reg.qbus_adr = TRUE;
	}
	else if (snap_reg.mjres || wire.mjres_rc || wire.sync_fedge)
	{
		reg.qbus_adr = FALSE;
	}

	reg.qbus_win_h = snap_reg.qbus_win;
	reg.sync_out_h = snap_reg.sync_out;
	reg.qbus_free_h = !(wire.pin_dmr_in || wire.pin_sack_in || (wire.pin_pa != 0));
	reg.qbus_nosr_h = wire.qbus_nosr_rc;

	if (wire.qbus_done)
	{
		reg.qbus_own_rp = FALSE;
	}
	else if (snap_reg.dmr_req_l && snap_reg.qbus_gnt_l)
	{
		reg.qbus_own_rp = TRUE;
	}

	reg.qbus_flag = wire.qbus_flag_rc;
	reg.qbus_tovf = ((snap_reg.qbus_timer >> 2) & 0xf) == 0xf;
	//
	// Error register at 177704
	// Bit 0 is Double Error
	// Bit 1 is Undefine Opcode
	// Bit 5 is not used (always read as one)
	// Bit 6 is Odd Address Trap (is not implmented)
	//
	if (wire.reset_rc || snap_reg.ir_set)
	{
		reg.reg_err[0] = FALSE;
		reg.reg_err[1] = FALSE;
		reg.reg_err[2] = FALSE;
		reg.reg_err[3] = FALSE;
		reg.reg_err[4] = FALSE;
		reg.reg_err[5] = TRUE;
		reg.reg_err[6] = FALSE;
		reg.reg_err[7] = FALSE;
	}
	else
	{
		//
		// Bit 0 - double error
		//
		if ((snap_reg.exc_dbl[0] && wire.exc_uop)
		 || (snap_reg.exc_dbl[1] && snap_reg.qbus_tovf)
		 || (snap_reg.exc_dbl[2] && wire.exc_oat)
		 || (snap_reg.exc_dbl[3] && wire.exc_err7))
		{
			reg.reg_err[0] = TRUE;
		}
		else if (wire.sel_out && snap_reg.sel_04)
		{
			reg.reg_err[0] = wire.wb_pio_dat_i[0];
		}
		//
		// Bit 1 - undefined opcode
		//
		if (wire.exc_uop)
		{
			reg.reg_err[1] = TRUE;
		}
		else if (wire.sel_out && snap_reg.sel_04)
		{
			reg.reg_err[1] = wire.wb_pio_dat_i[1];
		}
		//
		// Bit 2 && 3 - unknown error, exceptions not implemented
		//
		if (wire.exc_err2)
		{
			reg.reg_err[2] = TRUE;
		}
		else if (wire.sel_out && snap_reg.sel_04)
		{
			reg.reg_err[2] = wire.wb_pio_dat_i[2];
		}

		if (wire.exc_err3)
		{
			reg.reg_err[3] = TRUE;
		}
		else if (wire.sel_out && snap_reg.sel_04)
		{
			reg.reg_err[3] = wire.wb_pio_dat_i[3];
		}
		//
		// Bit 4 - qbus timeout
		//
		if (snap_reg.qbus_tovf)
		{
			reg.reg_err[4] = TRUE;
		}
		else if (wire.sel_out && snap_reg.sel_04)
		{
			reg.reg_err[4] = wire.wb_pio_dat_i[4];
		}
		//
		// Bit 6 - odd address trap
		//
		if (wire.exc_oat)
		{
			reg.reg_err[6] = TRUE;
		}
		else if (wire.sel_out && snap_reg.sel_04)
		{
			reg.reg_err[6] = wire.wb_pio_dat_i[6];
		}
		//
		// Bit 7 - unknown error, no exception
		//
		if (wire.exc_err7)
		{
			reg.reg_err[7] = TRUE;
		}
		else if (wire.sel_out && snap_reg.sel_04)
		{
			reg.reg_err[7] = wire.wb_pio_dat_i[7];
		}
	}
	//
	// Double error detectors
	//
	if (!wire.exc_uop)
	{
		reg.exc_dbl[0] = snap_reg.reg_err[1];
	}
	if (!snap_reg.qbus_tovf)
	{
		reg.exc_dbl[1] = snap_reg.reg_err[4];
	}
	if (!wire.exc_oat)
	{
		reg.exc_dbl[2] = snap_reg.reg_err[6];
	}
	if (!wire.exc_err7)
	{
		reg.exc_dbl[3] = snap_reg.reg_err[7];
	}

	if (!wire.pin_sync_in)
	{
		int n = ((wire.pin_ad_in & 0xf) >> 1);
		BOOL sel177x = ((wire.pin_ad_in & 0177700) == 0177700) && (((wire.pin_ad_in >> 4) & 3) == wire.pin_pa);

		reg.sel_xx = sel177x;
		reg.sel_00 = sel177x && (n == 0);
		reg.sel_02 = sel177x && (n == 1);
		reg.sel_04 = sel177x && (n == 2);
		reg.sel_06 = sel177x && (n == 3);
		reg.sel_10 = sel177x && (n == 4);
		reg.sel_12 = sel177x && (n == 5);
		reg.sel_14 = sel177x && (n == 6);
		reg.sel_16 = sel177x && (n == 7);
	}

	//
	// Original circuit contains error
	// The start_irq request is reset by interrupt ack and EMT execution (last one is wrong)
	//
	WORD nVsel;
	bita2data(snap_reg.vsel, nVsel, 4);
	if (snap_reg.reset || (((nVsel & 7) == 6) && (((snap_reg.nPLR >> 25) & 0xf) == 0x2) && ((snap_reg.plr[13] && !snap_reg.plr[14]) || snap_reg.plr[11])))
	{
		reg.start_irq = FALSE;
	}
	else if (wire.rply_s1)
	{
		reg.start_irq = TRUE;
	}
	//______________________________________________________________________________
	//
	// Interrupt controller
	//
	reg.rq[0] = snap_reg.psw[10];
	reg.rq[1] = snap_reg.plir[0];   // pli4r
	reg.rq[2] = snap_reg.psw[11];
	reg.rq[3] = snap_reg.uop;
	reg.rq[4] = snap_reg.psw[7];
	reg.rq[9] = snap_reg.qbto;
	reg.rq[10] = snap_reg.reg_err[0];
	reg.rq[11] = wire.pin_aclo && snap_reg.aclo;
	reg.rq[12] = snap_reg.reg_csr[2];
	reg.rq[13] = !wire.pin_aclo && snap_reg.acok;
	reg.rq[14] = wire.pin_irq[1];
	reg.rq[15] = snap_reg.psw[4];
	reg.rq[16] = wire.pin_irq[2] && snap_reg.irq2;
	reg.rq[17] = snap_reg.ivto;
	reg.rq[18] = wire.pin_irq[3] && snap_reg.irq3;
	//
	// Only master CPU processes vectored interrupts
	// Matrix accepts low level as active (asserted request)
	//
	reg.rq[8] = !(wire.pin_irq[0] && (wire.pin_pa == 0));
	//
	// Not used matrix inputs
	//
	reg.rq[5] = FALSE;
	reg.rq[6] = FALSE;
	reg.rq[7] = FALSE;
	//
	// VE timer interrupt
	//
	reg.rq[19] = wire.tve_irq;

	if (wire.pli_stb)
	{
		reg.plir[1] = wire.pli[6];
		reg.plir[2] = wire.pli[8];
		reg.plir[3] = wire.pli[10];
	}
	//
	// ACLO edge detectors
	//
	if (snap_reg.reset || wire.aclo_ack)
	{
		reg.aclo = FALSE;
	}
	else if (!wire.pin_aclo)
	{
		reg.aclo = TRUE;
	}

	if (wire.pin_dclo || wire.aclo_ack)
	{
		reg.acok = FALSE;
	}
	else if (wire.pin_aclo)
	{
		reg.acok = TRUE;
	}
	//
	// IRQ2 and IRQ3 falling edge detectors
	// Also resettable by internal INIT
	//
	if (wire.pin_init_out || wire.pin_init_in)
	{
		reg.irq2 = FALSE;
		reg.irq3 = FALSE;
	}
	else
	{
		if (wire.irq2_ack)
		{
			reg.irq2 = FALSE;
		}
		else if (!wire.pin_irq[2])
		{
			reg.irq2 = TRUE;
		}

		if (wire.irq3_ack)
		{
			reg.irq3 = FALSE;
		}
		else if (!wire.pin_irq[3])
		{
			reg.irq3 = TRUE;
		}
	}
	//
	// Error exception latches
	//
	if (snap_reg.reset || wire.uop_ack)
	{
		reg.uop = FALSE;
		reg.qbto = FALSE;
		reg.ivto = FALSE;
	}
	else
	{
		if (wire.exc_uop)
		{
			reg.uop = TRUE;
		}
		if (snap_reg.qbus_tovf && wire.iako_out)
		{
			reg.ivto = TRUE;
		}
		if (snap_reg.qbus_tovf || wire.exc_oat)
		{
			reg.qbto = TRUE;
		}
	}

	reg.pli_req = wire.pli_req_rc;

	if (wire.pli_stb)
	{
		reg.vsel[3] = wire.pli[3];
		reg.vsel[2] = !wire.pli[2];
		reg.vsel[1] = wire.pli[1];
		reg.vsel[0] = wire.pli[0];
	}
	else if (snap_reg.plm1x)
	{
		reg.vsel[3] = snap_reg.plr[18];
		reg.vsel[2] = !snap_reg.plr[20];
		reg.vsel[1] = !snap_reg.plr[21];
		reg.vsel[0] = !snap_reg.plr[22];
	}
	//______________________________________________________________________________
	//
	// Reset circuits
	//
	reg.reset = wire.reset_rc;
	reg.mjres = wire.mjres_rc;
	reg.mjres_h = snap_reg.mjres;
	reg.abort = wire.abort_rc;
	reg.abort_tm = snap_reg.abort && !wire.reset_rc;

	if (wire.reset_rc)
	{
		reg.init_out[0] = FALSE;
	}
	else if (snap_reg.plm1x_hl)
	{
		if (!snap_reg.plr[23])
		{
			reg.init_out[0] = FALSE;
		}
		else if (!snap_reg.plr[10])
		{
			reg.init_out[0] = TRUE;
		}
	}

	reg.init_out[1] = snap_reg.init_out[0];
	reg.init_out[2] = snap_reg.init_out[1];
	//______________________________________________________________________________
	//
	// Microcode state machine
	//
	//
	// Instruction register
	//
	if (wire.ir_stb1)
	{
		reg.ira = wire.pin_ad_in;
	}

	if (snap_reg.ir_stb2)
	{
		reg.ir = (wire.ir_stb1) ? wire.pin_ad_in : snap_reg.ira;
	}

	//
	// Other bits have no reset facility
	//
	if (wire.plm_stb_rc)
	{
		//
		// Some of plm is not used here (microcode address directly uses plx instead)
		//
		for (int i = 0; i < 34; ++i)
		{
			reg.plm[i] = wire.plx[i];
		}
		bita2data(reg.plm, reg.nPLM, 34);

	}
	// Microcode register
	//
	if (snap_reg.ustb1_hl)
	{
		if (snap_reg.mj_stb1)
		{
			reg.mj[11] = snap_reg.psw[4];
			if (snap_reg.psw_mj)
			{
				reg.mj[10] = snap_reg.psw[3];
				reg.mj[9]  = snap_reg.psw[2];
				reg.mj[8]  = snap_reg.psw[1];
			}
			else
			{
				reg.mj[10] = snap_reg.freg[3];
				reg.mj[9]  = snap_reg.freg[2];
				reg.mj[8]  = snap_reg.freg[1];
			}
		}

		if (snap_reg.mj_stb2)
		{
			reg.mj[7] = snap_reg.psw_mj ? snap_reg.psw[0] : snap_reg.freg[0];
		}
	}
	//
	// ALU
	//
	//
	// Transaction address low bit latch
	//
	if (wire.sync_stb)
	{
		reg.au_ta0 = GETBIT(snap_reg.areg, 0) && snap_reg.plrt[6];
	}
}

void C1801VM1::always_posedge_pin_clk_n_or_posedge_mjres_part_mjres()
{
	//______________________________________________________________________________
	//
	// Control and glue logic
	//
	reg.ustb = FALSE;
	reg.alu_busy_fp = FALSE;
	reg.tplmz[0] = FALSE;
	reg.tplmz[1] = FALSE;
	reg.tplmz[2] = FALSE;
	reg.tplmz[3] = FALSE;
	//______________________________________________________________________________
	//
	// Qbus logic
	//
	reg.qbus_aseq[0] = FALSE;
	reg.qbus_aseq[1] = FALSE;
}

void C1801VM1::always_posedge_pin_clk_n_or_posedge_mjres_part_pin_clk_n()
{
	//______________________________________________________________________________
	//
	// Control and glue logic
	//
	if (snap_reg.ustb_h)
	{
		reg.ustb = FALSE;
	}
	else if ((snap_reg.alu_busy_rp || (snap_reg.alu_busy_fp && !snap_reg.ustb1_h)) && wire.alu_qrdy)
	{
		reg.ustb = TRUE;
	}

	reg.alu_busy_fp = snap_reg.alu_busy_rp;

	if (wire.plm_ena_fc)
	{
		int n = ((snap_reg.nPLM & 0xf) >> 1);
		reg.tplmz[1] = (n == 4);
		reg.tplmz[2] = (n == 2);
		reg.tplmz[3] = (n == 3);
	}
	//______________________________________________________________________________
	//
	// Qbus logic
	//
	//
	// Reset the completed request if it is only one
	//
	if (wire.au_astb)
	{
		reg.qbus_aseq[0] = TRUE;
	}
	else if (snap_reg.oe_clr && !snap_reg.qbus_aseq[1])
	{
		reg.qbus_aseq[0] = FALSE;
	}
	//
	// The third request is impossible - the state machine
	// is suspended by qbus_nrdy
	//
	if (snap_reg.oe_clr)
	{
		reg.qbus_aseq[1] = FALSE;
	}
	else if (wire.au_astb && snap_reg.qbus_aseq[0])
	{
		reg.qbus_aseq[1] = TRUE;
	}

}

void C1801VM1::always_posedge_pin_clk_p_or_posedge_mjres_part_mjres()
{
	//______________________________________________________________________________
	//
	// Control and glue logic
	//
	reg.ustb_h = FALSE;
	reg.alu_busy_rp = FALSE;
	reg.alu_nrdy = FALSE;
	reg.ir_stb2 = FALSE;
	reg.ir_stop = FALSE;
	//______________________________________________________________________________
	//
	// Qbus logic
	//
	reg.dmr_req = FALSE;
	reg.dout_req = FALSE;
	reg.dout_out = FALSE;
	reg.din_out = FALSE;
	reg.sack_out = FALSE;
	reg.qbus_req = FALSE;
	//______________________________________________________________________________
	//
	// Interrupt controller
	//
	reg.iak_flag = FALSE;
	//______________________________________________________________________________
	//
	// Microcode register
	//
	//
	// Least bits are produced directly from main matrix
	// and have the async reset feature (next microcode address)
	//
	reg.mj[6] = TRUE;
	reg.mj[5] = TRUE;
	reg.mj[4] = TRUE;
	reg.mj[3] = TRUE;
	reg.mj[2] = TRUE;
	reg.mj[1] = TRUE;
	//
	// Bits 14:12 poll the interrupt controller state
	//
	reg.mj[14] = FALSE;
	reg.mj[13] = FALSE;
	reg.mj[12] = FALSE;
}
void C1801VM1::always_posedge_pin_clk_p_or_posedge_mjres_part_pin_clk_p()
{
	//______________________________________________________________________________
	//
	// Control and glue logic
	//
	reg.ustb_h = snap_reg.ustb;

	if (snap_reg.plm2x)
	{
		reg.alu_busy_rp = TRUE;
	}
	else if (snap_reg.ustb1)
	{
		reg.alu_busy_rp = FALSE;
	}

	if (snap_reg.plm2x)
	{
		reg.alu_nrdy = TRUE;
	}
	else if (snap_reg.ustb1_h)
	{
		reg.alu_nrdy = FALSE;
	}

	if (wire.ir_clr)
	{
		reg.ir_stb2 = FALSE;
	}
	else if (snap_reg.ir_set)
	{
		reg.ir_stb2 = TRUE;
	}

	if (!snap_reg.reg_csr[0])
	{
		reg.ir_stop = FALSE;
	}
	else if (wire.ir_clr)
	{
		reg.ir_stop = TRUE;
	}
	//______________________________________________________________________________
	//
	// Qbus logic
	//
	reg.dmr_req = wire.dmr_req_rc;
	reg.dout_req = wire.dout_req_rc;

	if (snap_reg.dout_done)
	{
		reg.dout_out = FALSE;
	}
	else if (wire.dout_start)
	{
		reg.dout_out = TRUE;
	}

	if (wire.din_done)
	{
		reg.din_out = FALSE;
	}
	else if (wire.din_start)
	{
		reg.din_out = TRUE;
	}

	if (wire.qbus_done)
	{
		reg.sack_out = FALSE;
	}
	else if (wire.pin_dmgi && (wire.qbus_nosr_rc || (snap_reg.sync_out && !snap_reg.qbus_win)) && snap_reg.qbus_req)
	{
		reg.sack_out = TRUE;
	}

	if (wire.qbus_done)
	{
		reg.qbus_req = FALSE;
	}
	else if (wire.dmr_out && !wire.pin_dmgi)
	{
		reg.qbus_req = TRUE;
	}
	//______________________________________________________________________________
	//
	// Interrupt controller
	//
	WORD nVsel;
	bita2data(snap_reg.vsel, nVsel, 4);
	if (wire.din_done)
	{
		reg.iak_flag = FALSE;
	}
	else if (snap_reg.plm_ena && (((snap_reg.nPLR >> 25) & 0xf) == 2) && (snap_reg.plr[11] || (snap_reg.plr[13] && !snap_reg.plr[14])) && (nVsel == 017))
	{
		reg.iak_flag = TRUE;
	}
	//______________________________________________________________________________
	//
	// Microcode register
	//
	//
	// Least bits are produced directly from main matrix
	// and have the async reset feature (next microcode address)
	//
	if (wire.plm_stb_rc)
	{
		reg.mj[1] = !wire.plx[24];
		reg.mj[2] = !wire.plx[19];
		reg.mj[3] = wire.plx[15];
		reg.mj[4] = wire.plx[9];
		reg.mj[5] = wire.plx[5];
		reg.mj[6] = !wire.plx[0];
	}
	//
	// Bits 14:12 poll the interrupt controller state
	//
	if (wire.pli_stb)
	{
		reg.mj[14] = !wire.pli[5];
		reg.mj[13] = !wire.pli[7];
		reg.mj[12] = wire.pli[9];
	}
	else if (snap_reg.plm1x)
	{
		reg.mj[14] = !snap_reg.plr[14];
		reg.mj[13] = snap_reg.plr[16];
		reg.mj[12] = !snap_reg.plr[17];
	}

}

//
// Qbus timeout counter
//
// original 1801BM1 timer has 1/8 input prescaler which is no reset and
// provides bus timeout in a range 56..63 processor clocks. This model
// emulates the average value 60 clocks.
//
// Timeout exception request should be synchronized with raising clk's edge
//
void C1801VM1::always_posedge_pin_clk_n_or_negedge_qbus_tena()
{
	if (!wire.qbus_tena)
	{
		reg.qbus_timer = 0;
	}
	else
	{
		if (!snap_reg.qbus_tovf)
		{
			reg.qbus_timer = (snap_reg.qbus_timer + 1) & 077;
		}
	}
}

void C1801VM1::always_posedge_pin_clk_p_or_posedge_reset_part_reset()
{
	//______________________________________________________________________________
	//
	// Qbus logic
	//
	//
	// Control register at 177700
	//
	reg.reg_csr[0] = FALSE;
	reg.reg_csr[1] = FALSE;
	reg.reg_csr[2] = FALSE;
	//______________________________________________________________________________
	//
	// Interrupt controller
	//
	reg.plir[0] = FALSE;
}
void C1801VM1::always_posedge_pin_clk_p_or_posedge_reset_part_pin_clk_p()
{
	//______________________________________________________________________________
	//
	// Qbus logic
	//
	//
	// Control register at 177700
	//
	//
	// Bit 0 of control register
	//
	if ((snap_reg.plm1x_hl && !snap_reg.plr[25]) || (snap_reg.ir_set && snap_reg.reg_csr[1]))
	{
		reg.reg_csr[0] = TRUE;
	}
	else if (wire.sel_out && snap_reg.sel_00)
	{
		reg.reg_csr[0] = wire.wb_pio_dat_i[0];
	}
	//
	// Bit 1 of control register (controls set of bit 0)
	//
	if (wire.sel_out && snap_reg.sel_00)
	{
		reg.reg_csr[1] = wire.wb_pio_dat_i[1];
	}
	//
	// Bit 2 of control register (wait state)
	//
	if (snap_reg.plm1x_hl && !snap_reg.plr[27])
	{
		reg.reg_csr[2] = FALSE;
	}
	else if (snap_reg.plm1x_hl && !snap_reg.plr[7])
	{
		reg.reg_csr[2] = TRUE;
	}
	else if (wire.sel_out && snap_reg.sel_00)
	{
		reg.reg_csr[2] = wire.wb_pio_dat_i[2];
	}
	//______________________________________________________________________________
	//
	// Interrupt controller
	//
	if (wire.pli_stb)
	{
		reg.plir[0] = wire.pli[4];
	}

}

void C1801VM1::always_posedge_pin_clk_n_or_posedge_reset_part_reset()
{
	for (int i = 0; i < 16; ++i)
	{
		reg.psw[i] = FALSE;
	}
	reg.psw[8] = ((wire.pin_pa & 1) == 1);
	reg.psw[9] = ((wire.pin_pa & 2) == 2);
}

void C1801VM1::always_posedge_pin_clk_n_or_posedge_reset_part_pin_clk_n()
{
	//
	// ALU result flags and PSW
	//
	//
	// Start microcode correctly sets the PSW
	// But simulator can not calculate ALU function
	// correctly if PSW in initially undefined
	//
	// The PSW is written by microcode with 340 value
	// while executing reset sequence
	//
	if (snap_reg.au_alsl && wire.au_pstbx)
	{
		BOOL bX[16];
		data2bita(wire.x, bX, 16);
		for (int i = 0; i < 8; ++i)
		{
			reg.psw[i] = bX[i];
		}
	}
	else
	{
		//
		// Latch the ALU flag results immediately after ALU completion
		//
		if (snap_reg.psw_stbc && snap_reg.au_alsl)
		{
			reg.psw[0] = wire.flag[0];
		}

		if (snap_reg.psw_stb && snap_reg.au_alsl)
		{
			reg.psw[1] = wire.flag[1];
			reg.psw[2] = wire.flag[2];
			reg.psw[3] = wire.flag[3];
		}
	}

	if (snap_reg.au_alsh && wire.au_pstbx)
	{
		reg.psw[8] = ((wire.pin_pa & 1) == 1);
		reg.psw[9] = ((wire.pin_pa & 2) == 2);
		BOOL bX[16];
		data2bita(wire.x, bX, 16);
		for (int i = 10; i < 16; ++i)
		{
			reg.psw[i] = bX[i];
		}
	}

}

void C1801VM1::always_posedge_pin_clk_p_or_posedge_abort_part_abort()
{
	reg.mj[0] = FALSE;
}

void C1801VM1::always_posedge_pin_clk_p_or_posedge_abort_part_pin_clk_p()
{
	//______________________________________________________________________________
	//
	// Microcode register
	//
	//
	// The least bit is reset by abort
	// and set by common reset
	//
	if (snap_reg.reset)
	{
		reg.mj[0] = TRUE;
	}
	else if (wire.plm_stb_rc)
	{
		reg.mj[0] = !wire.plx[29];
	}

}

//
// On schematics vc_vreg is {pli[3], !pli[2], pli[1], pli[0]}
//
void C1801VM1::always_a()
{
	BYTE nVcVreg;
	bita2data(snap_reg.vc_vreg, nVcVreg, 4);
	nVcVreg &= 0xf;
	static WORD pVMuxData[16] = {
		0160006,	//0	// double error
		0000020,	//1	// IOT instruction
		0000010,	//2	// reserved opcode
		0000014,	//3	// T-bit/BPT trap
		0000004,	//4	// invalid opcode
		0177716,	//5	// or qbus timeout
		0000030,	//6	// EMT instruction
		0160012,	//7	// int ack timeout
		0000270,	//8	// IRQ3 falling edge
		0000024,	//9	// ACLO falling edge
		0000100,	//10// IRQ2 falling edge
		0160002,	//11// IRQ1 low level/HALT
		0000034,	//12// TRAP instruction
		0,			//13//instruction register
		0,			//14 //start @177704
		0			//15 //unused vector (iako)
	};
	switch (nVcVreg)
	{
	case 5:  reg.vmux = pVMuxData[nVcVreg] | (wire.pin_pa << 4);	// qbus timeout
		break;
	case 13: reg.vmux = snap_reg.ir;		// instruction register
		break;
	default: reg.vmux = pVMuxData[nVcVreg];					//
	}

	//
	// Constant generator and opcode fields shifter/truncator
	//
	// On schematics csel is plr[28:25]
	//
	//    csel0 = 4b'00xx
	//    csel1 = 4b'01xx
	//    csel2 = 4b'10xx
	//    csel3 = 4b'11xx
	//    csel4 = 4b'xx00
	//    csel5 = 4b'xx01
	//    csel6 = 4b'xx10
	//    csel7 = 4b'xx11
	//
	
	int nx = (snap_reg.nPLR >> 25) & 0xf;
	static WORD pCMuxData[16] = {
		0,			//0	// csel0 && csel4 (CLx/SEx)
		0000340,	//1	// csel0 && csel5 (start PSW constant)
		0000000,	//2	// csel0 && csel6 (vector output)
		0000002,	//3	// csel0 && csel7
		0,			//4	// csel1 && csel4 (MARK/SOB)
		0177716,	//5	// csel1 && csel5
		0177777,	//6	// csel1 && csel6
		0000001,	//7	// csel1 && csel7
		0,			//8	// csel2 && csel4
		0100000,	//9	// csel2 && csel5
		0177676,	//10// csel2 && csel6
		0000020,	//11// csel2 && csel7 (MUL)
		0,			//12// csel3 && csel4 (ADC/SBC)
		0177400,	//13// csel3 && csel5 (start address AND)
		0000010,	//14// csel3 && csel6
		0000000		//15// csel3 && csel7 (vector read/vsel)
	};
	switch (nx)
	{
	case 0:	reg.cmux = snap_reg.ir & 0xf;			// csel0 && csel4 (CLx/SEx)
		break;
	case 4: reg.cmux = (snap_reg.ir & 077) << 1;	// csel1 && csel4 (MARK/SOB)
		break;
	case 8:											//
	{												// csel2 && csel4
		if (GETBIT(snap_reg.ir, 7))								//
			reg.cmux = 0177000 | (snap_reg.ir & 0377) << 1;		// low byte x2   (BR/Bxx)
		else										// sign extension
			reg.cmux = (snap_reg.ir & 0377) << 1;	// for branches
	}												//
	break;
	case 12: reg.cmux = snap_reg.psw[0] ? 1 : 0;	// csel3 && csel4 (ADC/SBC)
		break;
	default: reg.cmux = pCMuxData[nx];
	}
}

#ifdef DBG_OUT_PLI_LOG

void C1801VM1::OutPLILog()
{
	static int nOutclklog = 0;
	if (m_nClk != 0)
	{
		if (m_nClk == nOutclklog) //  .
		{
			return;
		}
		else
		{
			nOutclklog = m_nClk;
		}
	}

	fprintf(m_PLILog, "%06d  ", m_nClk);

	for (int i = 19; i >= 0; i--)
	{
		fprintf(m_PLILog, "      %d", reg.rq[i]);
	}
	fprintf(m_PLILog, "       ");
	for (int i = 10; i >= 0; i--)
	{
		fprintf(m_PLILog, "      %d", wire.pli[i]);
	}
	fprintf(m_PLILog, "\n");
}
#endif
#ifdef DBG_OUT_PLM_LOG
static char *strOptypes[8] = {
	"nothing",
	"nothing",
	"write word",
	"write byte",
	"read word",
	"read byte",
	"read - modify - write word",
	"read - modify - write byte"
};

void C1801VM1::OutPLMLog()
{
	static int nOutclklog = 0;
	if (m_nClk != 0)
	{
		if (m_nClk == nOutclklog) //  .
		{
			return;
		}
		else
		{
			nOutclklog = m_nClk;
		}
	}
	fprintf(m_PLMLog, "%06d  ", m_nClk);
	fprintf(m_PLMLog, "%06o  ", reg.ir);
	fprintf(m_PLMLog, "%d", (reg.mj[14] ? 1 : 0));
	fprintf(m_PLMLog, "%d", (reg.mj[13] ? 1 : 0));
	fprintf(m_PLMLog, "%d ", (reg.mj[12] ? 1 : 0));
	fprintf(m_PLMLog, "%d ", (reg.mj[11] ? 1 : 0));
	fprintf(m_PLMLog, "%d ", (reg.mj[10] ? 1 : 0));
	fprintf(m_PLMLog, "%d ", (reg.mj[9] ? 1 : 0));
	fprintf(m_PLMLog, "%d ", (reg.mj[8] ? 1 : 0));
	fprintf(m_PLMLog, "%d ", (reg.mj[7] ? 1 : 0));
	WORD nMJ;
	bita2data(reg.mj, nMJ, 15);
	fprintf(m_PLMLog, " 0x%02x    ", nMJ & 0x7f);
	for (int i = 33; i >= 0; i--)
	{
		fprintf(m_PLMLog, "      %d", wire.plx[i]);
	}
	__int64 nPLX;
	bita2data(wire.plx, nPLX, 34);
	int Optype = (nPLX >> 6) & 7;
	//    plx[0]   ~mj[6];
	//    plx[5]    mj[5]
	//    plx[9]    mj[4]
	//    plx[12]   opcode is not recognized
	//    plx[15]   mj[3]
	//    plx[19]  ~mj[2]
	//    plx[24]  ~mj[1]
	//    plx[29]  ~mj[0]
	int nextPLMOP
		= (wire.plx[0] ? 0 : 0x40)
		| (wire.plx[5] ? 0x20 : 0)
		| (wire.plx[9] ? 0x10 : 0)
		| (wire.plx[15] ? 0x8 : 0)
		| (wire.plx[19] ? 0 : 4)
		| (wire.plx[24] ? 0 : 2)
		| (wire.plx[29] ? 0 : 1);

	fprintf(m_PLMLog, " next:0x%02x / %s", nextPLMOP, strOptypes[Optype]);

	fprintf(m_PLMLog, "\n");
	fflush(m_PLMLog);
}


#endif
